/***************************************************************************
 * Copyright 2012 BMW Car IT GmbH
 * Copyright (C) 2012 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "SignalHandler.h"
#include "Log.h"
#include "config.h"

#include <signal.h>
#include <stdlib.h>   // exit
#include <pthread.h>  // mutex

#ifdef HAVE_EXECINFO_H
#include <execinfo.h>
#endif


//===========================================================================
// global variables
//===========================================================================
static pthread_mutex_t gShutdownReceived = PTHREAD_MUTEX_INITIALIZER;
static int last_signal;
#ifdef HAVE_BACKTRACE
    const int maxStackSize = 64;
    void* stack[maxStackSize];
    size_t count;
#endif
//===========================================================================
// Prototypes
//===========================================================================
void incomingSignalCallback(int sig);
void getStackTrace();
void printStackTrace();

//===========================================================================
// class implementation
//===========================================================================
SignalHandler::SignalHandler()
{
    LOG_DEBUG("LayerManagerService", "Enabled signal handling.");
    pthread_mutex_lock(&gShutdownReceived); // no shutdown event received
    last_signal = 0;
    signal(SIGTERM, incomingSignalCallback);
    signal(SIGINT, incomingSignalCallback);
    signal(SIGABRT, incomingSignalCallback);
    signal(SIGUSR2, incomingSignalCallback);
}

SignalHandler::~SignalHandler()
{
    LOG_DEBUG("LayerManagerService", "Disabled signal handling.");
    signal(SIGTERM, SIG_DFL);
    signal(SIGINT, SIG_DFL);
    signal(SIGABRT, SIG_DFL);
    signal(SIGUSR2, SIG_DFL);
}

void SignalHandler::waitForShutdownSignal()
{
    // this mutex is unlocked by the signal handler callback routine
    // e.g. on incoming SIGTERM or SIGABRT

    pthread_mutex_lock(&gShutdownReceived);
	switch (last_signal)
    {
    case SIGTERM:
        LOG_WARNING("LayerManagerService",
                    "Signal SIGTERM received. Shutting down.");
        break;

    case SIGINT:
        LOG_WARNING("LayerManagerService",
                    "Signal SIGINT received. Shutting down.");
        break;

    case SIGABRT:
        LOG_ERROR("LayerManagerService",
                  "Signal SIGABRT received. Shutting down.");
        printStackTrace();
        break;

    default:
        LOG_WARNING("LayerManagerService", "Signal " << last_signal << " received.");
        break;
    }
    pthread_mutex_unlock(&gShutdownReceived);
}

//===========================================================================
// signal handling functions
//===========================================================================
void getStackTrace()
{
#ifdef HAVE_BACKTRACE
    count = backtrace(stack, maxStackSize);
#endif
}

void printStackTrace()
{
#ifdef HAVE_BACKTRACE
    char **lines = backtrace_symbols(stack, count);

    LOG_ERROR("LayerManagerService",
              "--------------------------------------------------");
    for (unsigned int i = 0; i < count; ++i)
    {
        LOG_ERROR("LayerManagerService",
                  "Stack-Trace [" << i << "]: " << lines[i]);
    }
    LOG_ERROR("LayerManagerService",
              "--------------------------------------------------");
#endif

    LOG_ERROR("LayerManagerService", "Exiting application.")
}

void incomingSignalCallback(int sig)
{
    last_signal = sig;
	switch (sig)
    {
    case SIGTERM:
    case SIGINT:
        pthread_mutex_unlock(&gShutdownReceived);
        break;

    case SIGABRT:
        getStackTrace();
        pthread_mutex_unlock(&gShutdownReceived);
        break;

    case SIGUSR2:
        LOG_WARNING("LayerManagerService",
                    "Signal SIGUSR2 received. "
                    "To print callstacks use exception handler ioctl IOCTL_EXCHND_ON_DEMAND, instead.");
                break;
    default:
        LOG_WARNING("LayerManagerService", "Signal " << sig << " received.");
        break;
    }
}
